from accounts.permissions.is_admin_or_analyst import IsAdminOrAnalyst
from rest_framework.response import Response
from rest_framework.views import APIView
from django.utils.timezone import now, timedelta
from django.core.paginator import Paginator
from django.db.models import Q
import logging
import requests
from rest_framework import status
from rest_framework.permissions import IsAuthenticated
from rest_framework.permissions import AllowAny

# Local imports
from .models import Product, ActiveScraper,ProductClick
from .serializers import ProductSerializer, ProductClickSerializer
from .tasks import scrape_jumia_task, scrape_konga_task, scrape_slot_task 
from   accounts.utils.activity_logger import log_activity, log_search




logger = logging.getLogger(__name__)

class SearchProductView(APIView):
    permission_classes = [AllowAny]
    PAGE_SIZE = 20
    EXPECTED_SITES = {'jumia', 'slot.ng', 'konga'}
    STALE_JOB_THRESHOLD_MINUTES = 60 
    # --- SYNCHRONOUS DATABASE HELPER METHODS ---
    def _get_active_scrapers_sync(self, query):
        """
        Synchronous helper for getting active scraper status, filtering out stale jobs.
        Stale jobs are considered 'active' but have been running for too long.
        """
        stale_time = now() - timedelta(minutes=self.STALE_JOB_THRESHOLD_MINUTES)
        
        # Only look for jobs marked RUNNING that started recently.
        active_jobs = ActiveScraper.objects.filter(
            query=query, 
            status='RUNNING',
            # Ignore jobs that started before the stale threshold
            start_time__gt=stale_time 
        ).values_list('site', flat=True)

        # Collect the IDs of the jobs that are old and stalled for potential logging/reporting
        stale_jobs = ActiveScraper.objects.filter(
            query=query,
            status='RUNNING',
            start_time__lte=stale_time
        ).values_list('site', flat=True)

        # Log stale jobs, but do NOT return them as "in progress"
        if stale_jobs:
            logger.warning(f"Found stale RUNNING jobs for query '{query}' on sites: {list(stale_jobs)}")

        return list(active_jobs.distinct())
    
    def _perform_initial_query_sync(self, q_objects, page_number):
        """Synchronous helper for querying and paginating."""
        results = Product.objects.filter(q_objects).order_by('id')
        existing_sites = set(results.values_list('site', flat=True).distinct())

        paginator = Paginator(results, self.PAGE_SIZE)
        try:
            page_obj = paginator.get_page(page_number)
            paginated_results = page_obj.object_list
        except Exception:
            paginated_results = []
            page_obj = None
            
        # Returning results is not needed for the sync path, but kept for consistency
        return results, existing_sites, paginator, page_obj, paginated_results

    # --- Task Management ---

    def _start_celery_task(self, query: str, site:str)-> str | None:
        """Starts a Celery task and returns its task ID (job_id)."""
        if site == 'jumia':
            task = scrape_jumia_task.delay(query)
        elif site == 'slot.ng':
            task = scrape_slot_task.delay(query)
        elif site == 'konga':
            task = scrape_konga_task.delay(query)
        else:
            return None # Unknown site
            
        logger.info(f"Queued Celery task for {site}, Query='{query}'. ID: {task.id}")
        return task.id

# ----------------------------------------------------------------------
# --- MAIN SYNCHRONOUS API METHOD ---
# ----------------------------------------------------------------------
    def get(self, request):
        SCRAPE_COOLDOWN_MINUTES = 60
        JUMIA_PRIORITY = 0    # Highest priority
        DEFAULT_PRIORITY = 5  # Normal priority
        query = request.GET.get('q')
        # Ensure conversion of query parameters happens before passing to thread
        try:
            page_number = int(request.GET.get('page', 1))
        except ValueError:
            page_number = 1
            
        site_filter = request.GET.get('site')

        if not query:
            return Response({"error": "Query is required."}, status=400)

        # 1. Prepare Q objects 
        normalized = query.replace("-", " ").replace("_", " ")
        words = [w.strip() for w in normalized.split() if w.strip()]
        q_objects = Q()
        for word in words:
            q_objects |= Q(name__icontains=word)
        if site_filter:
            q_objects |= Q(site__iexact=site_filter)

        # 2. Initial Data Fetch (Now synchronous)
        results, existing_sites, paginator, page_obj, paginated_results = \
            self._perform_initial_query_sync(q_objects, page_number)
        scraping_status = "no_scraping"
        # ........................
        recently_completed_sites = ActiveScraper.objects.filter(
        query=query, 
        status='COMPLETED', 
        end_time__gt=now() - timedelta(minutes=SCRAPE_COOLDOWN_MINUTES)
        ).values_list('site', flat=True)

        # 2. Determine which sites to scrape/queue
        missing_sites = self.EXPECTED_SITES - existing_sites
        active_scrapers_for_query = self._get_active_scrapers_sync(query)
        sites_in_progress = list(active_scrapers_for_query)

        # Sites to start are missing sites that are NOT running AND have NOT been recently completed.
        sites_to_start = missing_sites - set(active_scrapers_for_query) - set(recently_completed_sites)
        # ........................
        
        started_jobs = []
    
        if sites_to_start:
            # Queue the scraping job(s)
            for site in sites_to_start:
                job_id = self._start_celery_task(query, site)
                if job_id:
                    started_jobs.append(site)
            
            sites_in_progress.extend(started_jobs)
            
            if not existing_sites:
                scraping_status = "initial_scraping_queued"
            else:
                scraping_status = "expanding_search_queued"
    
        if query:
            log_activity(
                user=request.user if request.user.is_authenticated else None,
                action="search",
                description=f"Searched for '{query}'",
                request=request
            )
            log_search(request, query)
        return Response({
            "results": ProductSerializer(paginated_results, many=True).data,
            "page": page_obj.number if page_obj else page_number,
            "has_next": page_obj.has_next() if page_obj else False,
            "has_previous": page_obj.has_previous() if page_obj else False,
            "total_pages": paginator.num_pages if paginator.count else 0,
            "total_count": paginator.count,
            "scraping_status": scraping_status, # Indicates to the client if scraping is running
            "sites_available": sorted(list(existing_sites)),
            "sites_in_progress": sorted(list(set(sites_in_progress))),
            "debug_info": {
                "results_in_page": len(paginated_results),
                "active_scrapers_in_db": active_scrapers_for_query,
                "sites_queued_now": started_jobs,
                "missing_sites": sorted(list(missing_sites))
            }
        })
    

class ProductClickAPIView(APIView):
    """
    API endpoint to record product clicks.
    """
    authentication_classes = [] 
    permission_classes = [AllowAny]

    def post(self, request, format=None):
        data = request.data.copy()
        print(data)
        ip = self.get_client_ip(request)
        print(ip)
        data['country'] = self.get_country_from_ip(ip)
        
        serializer = ProductClickSerializer(data=data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def get_client_ip(self, request):
        """
        Retrieve client IP address from request headers.
        """
        x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
        if x_forwarded_for:
            ip = x_forwarded_for.split(',')[0].strip()
        else:
            ip = request.META.get('REMOTE_ADDR')
        return ip

    def get_country_from_ip(self, ip):
    # Local / private IPs
        if ip in ('127.0.0.1', '::1') or ip.startswith('192.') or ip.startswith('10.'):
            return 'Localhost'

        try:
            response = requests.get(
                f'https://ipapi.co/{ip}/country_name/',
                timeout=3
            )
            if response.status_code == 200 and response.text:
                return response.text.strip()
        except requests.RequestException:
            pass

        return 'Unknown'
    

class ProductClickListAPIView(APIView):
    permission_classes = [IsAuthenticated, IsAdminOrAnalyst]
    """
    API endpoint to fetch all product clicks.
    """

    def get(self, request, format=None):
        clicks = ProductClick.objects.all().order_by('-timestamp')  
        serializer = ProductClickSerializer(clicks, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)
