import logging
from celery import shared_task
from django.utils.timezone import now
from django.db import transaction
from .models import Product, ActiveScraper
from .jumia import scrape_jumia_batch
from .slot import scrape_slot_batch
from .konga import scrape_konga_batch


from channels.layers import get_channel_layer
from asgiref.sync import async_to_sync
import re

logger = logging.getLogger(__name__)


def safe_group_name(query: str) -> str:
    """Generate a safe group name for Channels (max 100 chars)."""
    # Replace all non-allowed characters with underscores
    safe = re.sub(r'[^\w\.-]', '_', query)
    # Limit length to 90 (to allow prefix)
    return f"scraper_updates_{safe[:90]}"


# --- Helper function for saving products (reusable) ---

def _save_products_to_db(results, site_name):
    """Saves a batch of scraped products atomically, including image, stock, and policy details."""
    saved_count = 0
    with transaction.atomic():
        for item in results:
            try:
                # Use URL as the unique identifier for get_or_create to prevent duplicates
                product, created = Product.objects.get_or_create(
                    url=item['url'],
                    defaults={
                        'name': item['name'],
                        'site': item['site'],
                        'price': item.get('price', 0),
                        'image_url': item.get('image_url', ''),
                        'stock': item.get('stock', True),
                        'return_policy': item.get('return_policy', 'Return within 7 days'),
                        'last_checked': now()
                    }
                )
                if created:
                    saved_count += 1
            except Exception as e:
                # Log the error but continue saving other products in the batch
                logger.error(f"Error saving {site_name} product: {e} | Product URL: {item.get('url', 'Unknown')}", exc_info=True)
    return saved_count

# --- Helper function to manage ActiveScraper entry ---

def start_active_scraper_job(task_instance, query, site):
    """
    Creates or updates the ActiveScraper record for a new task run, 
    ensuring uniqueness and setting start status.
    """
    unique_fields = {'query': query, 'site': site}
    update_fields = {
        'job_id': task_instance.request.id,
        'status': 'RUNNING',
        'start_time': now(),
        'end_time': None, # Ensure end_time is reset when the job starts
    }

    try:
        active_scraper_entry, created = ActiveScraper.objects.update_or_create(
            defaults=update_fields,
            **unique_fields
        )

        if created:
            logger.info(f"Created new ActiveScraper entry for Query: '{query}', Site: '{site}'")
        else:
            logger.info(f"Updated existing ActiveScraper entry for Query: '{query}', Site: '{site}' with new Job ID: {task_instance.request.id}")
            
        return active_scraper_entry

    except Exception as e:
        logger.error(f"Critical error managing ActiveScraper job for {query}-{site}: {e}", exc_info=True)
        raise # Re-raise to ensure task fails if DB connection is lost

channel_layer = get_channel_layer()
def send_scraper_update(query, site, status, message):
    """
    Sends a WebSocket message to all clients listening on this query group.
    """
    # group_name = f"scraper_updates_{query}"
    group_name = safe_group_name(query)

    async_to_sync(channel_layer.group_send)(
        group_name,
        {
            'type': 'scraper.update',
            'status': status,   # e.g., 'running', 'completed', 'failed'
            'site': site,
            'query': query,
            'message': message
        }
    )


# ----------------------------------------------------------------------
# --- Celery Tasks (The new background workers) ---
# ----------------------------------------------------------------------

@shared_task(bind=True)
def scrape_jumia_task(self, query):
    site = 'jumia'
    try:
        # 1. Update ActiveScraper status to RUNNING (using robust helper)
        start_active_scraper_job(self, query, site)
        
        logger.info(f"[Task {self.request.id}] Starting {site} scrape for query: {query}")

        # 🟡 Notify frontend that the scraper has started
        send_scraper_update(query, site, 'running', f"{site} scraper started for '{query}'")
        
        # 2. Scrape the data
        jumia_results = scrape_jumia_batch(query)
        
        # 3. Save the results (FIX: Removed the 'query' argument here)
        saved_count = _save_products_to_db(jumia_results, site) 
        logger.info(f"[Task {self.request.id}] {site} saved {saved_count} new products for '{query}'.")

        # ✅ Notify frontend that the scraper completed
        send_scraper_update(query, site, 'completed', f"✅ {site} scraping completed — {saved_count} products saved.")

        # 4. Update status to COMPLETED and set end_time
        ActiveScraper.objects.filter(job_id=self.request.id).update(status='COMPLETED', end_time=now())
        return f"{site}: {saved_count} items saved."

    except Exception as e:
        logger.error(f"[Task {self.request.id}] CRITICAL ERROR during {site} scraping: {e}", exc_info=True)
        # Update status to FAILED and set end_time
        ActiveScraper.objects.filter(job_id=self.request.id).update(status='FAILED', end_time=now())
        # 🔴 Notify frontend that the scraper failed
        send_scraper_update(query, site, 'failed', f"❌ {site} scraping failed: {str(e)}")
        raise # Re-raise to let Celery log the error


@shared_task(bind=True)
def scrape_slot_task(self, query):
    site = 'slot.ng'
    try:
        start_active_scraper_job(self, query, site)

        logger.info(f"[Task {self.request.id}] Starting {site} scrape for query: {query}")

        # 🟡 Notify frontend that the scraper has started
        send_scraper_update(query, site, 'running', f"{site} scraper started for '{query}'")
        
        slot_results = scrape_slot_batch(query)
        # FIX: Removed the 'query' argument here
        saved_count = _save_products_to_db(slot_results, site)
        
        logger.info(f"[Task {self.request.id}] {site} saved {saved_count} new products for '{query}'.")
 #      ✅ Notify frontend that the scraper completed
        send_scraper_update(query, site, 'completed', f"✅ {site} scraping completed — {saved_count} products saved.")
        # 4. Update status to COMPLETED and set end_time
        ActiveScraper.objects.filter(job_id=self.request.id).update(status='COMPLETED', end_time=now())
        return f"{site}: {saved_count} items saved."
        
    except Exception as e:
        logger.error(f"[Task {self.request.id}] CRITICAL ERROR during {site} scraping: {e}", exc_info=True)
        # Update status to FAILED and set end_time
        ActiveScraper.objects.filter(job_id=self.request.id).update(status='FAILED', end_time=now())
        # 🔴 Notify frontend that the scraper failed
        send_scraper_update(query, site, 'failed', f"❌ {site} scraping failed: {str(e)}")
        raise

@shared_task(bind=True)
def scrape_konga_task(self, query):
    site = 'konga'
    try:
        start_active_scraper_job(self, query, site)

        logger.info(f"[Task {self.request.id}] Starting {site} scrape for query: {query}")

         # 🟡 Notify frontend that the scraper has started
        send_scraper_update(query, site, 'running', f"{site} scraper started for '{query}'")

        konga_results = scrape_konga_batch(query)
        # FIX: Removed the 'query' argument here
        saved_count = _save_products_to_db(konga_results, site) 
        
        logger.info(f"[Task {self.request.id}] {site} saved {saved_count} new products for '{query}'.")

         # ✅ Notify frontend that the scraper completed
        send_scraper_update(query, site, 'completed', f"✅ {site} scraping completed — {saved_count} products saved.")
        
        # Update status to COMPLETED and set end_time
        ActiveScraper.objects.filter(job_id=self.request.id).update(status='COMPLETED', end_time=now())
        return f"{site}: {saved_count} items saved."
        
    except Exception as e:
        logger.error(f"[Task {self.request.id}] CRITICAL ERROR during {site} scraping: {e}", exc_info=True)
        # Update status to FAILED and set end_time
        ActiveScraper.objects.filter(job_id=self.request.id).update(status='FAILED', end_time=now())
        # 🔴 Notify frontend that the scraper failed
        send_scraper_update(query, site, 'failed', f"❌ {site} scraping failed: {str(e)}")
        raise


@shared_task
def cleanup_stale_jobs_task():
    """Celery Beat task for cleanup."""
    from django.utils.timezone import timedelta 
    # NOTE: Set this value carefully. It should be less than the view's STALE_JOB_THRESHOLD_MINUTES (60 min).
    STALE_TIMEOUT_MINUTES = 10 
    timeout = now() - timedelta(minutes=STALE_TIMEOUT_MINUTES)
    
    # Change RUNNING jobs that are too old to EXPIRED
    stale_count = ActiveScraper.objects.filter(
        status='RUNNING', 
        start_time__lt=timeout
    ).update(status='EXPIRED', end_time=now())
    
    if stale_count:
        logger.warning(f"Cleaned up {stale_count} stale scraper jobs.")
    return f"Cleanup complete: {stale_count} jobs expired."

