import requests
from bs4 import BeautifulSoup
from urllib.parse import urljoin, urlencode
import re
import time
import threading
from queue import Queue

BASE_URL = "https://slot.ng"

def clean_price(price_str):
    """Convert price string like '₦250,000' into an integer (250000)."""
    if not price_str:
        return None
    price_str = re.sub(r"[^\d]", "", price_str)  # keep only digits
    return int(price_str) if price_str.isdigit() else None

def scrape_product_details(url, headers):
    """Scrape product details from a Slot.ng product page."""
    response = requests.get(url, headers=headers, timeout=20)
    response.raise_for_status()
    soup = BeautifulSoup(response.text, "html.parser")

    # Title
    title_tag = soup.select_one("h1.page-title span")
    title = title_tag.get_text(strip=True) if title_tag else "Unknown"

    # Price
    price_tag = soup.select_one("div.product-rate-price span[data-price-amount]")
    price = price_tag.get("data-price-amount") if price_tag else None

    # Image (take the first image from MagicToolbox selectors)
    img_tag = soup.select_one("div.MagicToolboxContainer.selectorsLeft "
                              ".MagicToolboxSelectorsContainer a")
    image_url = img_tag.get("href") if img_tag else None

    return {
        "url": url,
        "name": title,
        "site": "Slot.ng",
        "price": clean_price(price),
        "image_url": image_url,
        "stock": True,  # still default unless we add an "Out of stock" check
        "return_policy": "Return within 7 days"
    }

def scrape_slot_batch(query, batch_size=5, max_products=100, batch_callback=None):
    """
    Scrape Slot.ng and return results in batches to reduce waiting time.
    
    Args:
        query: Search query
        batch_size: Number of products per batch
        max_products: Maximum products to scrape
        batch_callback: Function to call with each batch of products
    """
    start_time = time.time()
    headers = {
        "User-Agent": (
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) "
            "AppleWebKit/537.36 (KHTML, like Gecko) "
            "Chrome/119.0.0.0 Safari/537.36"
        )
    }

    # Build the search results URL
    params = {"cat": "", "q": query}
    search_url = f"{BASE_URL}/index.php/catalogsearch/result/?{urlencode(params)}"
    print(f"Scraping search results: {search_url}")

    try:
        response = requests.get(search_url, headers=headers, timeout=20)
        if response.status_code != 200:
            print("Failed to fetch Slot.ng search results.")
            if batch_callback:
                batch_callback([], completed=True, error="Failed to fetch results")
            return []
    except requests.RequestException as e:
        print(f"Request error: {e}")
        if batch_callback:
            batch_callback([], completed=True, error=str(e))
        return []

    soup = BeautifulSoup(response.text, "html.parser")
    product_links = []

    # Grab product links
    for a_tag in soup.select("a.product-item-link"):
        href = a_tag.get("href")
        if href:
            full_url = urljoin(BASE_URL, href)
            if full_url not in product_links:
                product_links.append(full_url)
        if len(product_links) >= max_products:
            break

    print(f"Found {len(product_links)} products on Slot.ng")
    
    all_products = []

    # Process in batches
    for i in range(0, len(product_links), batch_size):
        batch_links = product_links[i:i + batch_size]
        batch_products = []
        
        for link in batch_links:
            try:
                details = scrape_product_details(link, headers)
                batch_products.append(details)
            except Exception as e:
                print(f"❌ Failed to scrape {link}: {e}")
                continue

        all_products.extend(batch_products)
        
        # Send batch to callback if provided
        if batch_callback:
            is_completed = (i + batch_size >= len(product_links))
            batch_callback(batch_products, completed=is_completed)
        
        # Small delay to simulate progressive loading
        time.sleep(0.1)

    end_time = time.time()
    print(f"✨ Slot.ng batch scraping finished in {end_time - start_time:.2f} seconds. Found {len(all_products)} products.")
    
    return all_products

def scrape_slot_async(query, batch_size=10, max_products=100):
    """
    Non-blocking version that uses threading to return batches immediately.
    Returns a Queue that yields batches of products.
    """
    def scraper_thread(query, batch_size, max_products, queue):
        try:
            def batch_handler(batch, completed=False, error=None):
                queue.put({
                    'batch': batch,
                    'completed': completed,
                    'error': error
                })
            
            scrape_slot_batch(query, batch_size, max_products, batch_handler)
        except Exception as e:
            queue.put({
                'batch': [],
                'completed': True,
                'error': str(e)
            })
    
    queue = Queue()
    thread = threading.Thread(
        target=scraper_thread, 
        args=(query, batch_size, max_products, queue),
        daemon=True
    )
    thread.start()
    
    return queue

# Original function maintained for backward compatibility
def scrape_slot(query, limit=100):
    """Original synchronous version - returns all products at once"""
    return scrape_slot_batch(query, batch_size=100, max_products=limit)