# products/models.py

from django.db import models
from django.utils.timezone import now

class Product(models.Model):
    name = models.CharField(max_length=191)
    site = models.CharField(max_length=50)  # e.g., 'jumia'
    price = models.DecimalField(max_digits=12, decimal_places=2)
    url = models.URLField(max_length=255,unique=True)
    image_url = models.URLField(null=True, blank=True,max_length=500)
    last_updated = models.DateTimeField(auto_now=True)
    stock = models.BooleanField(default=True)
    return_policy = models.TextField(null=True, blank=True)
    last_checked = models.DateTimeField(default=now)


    # class Meta:
    #     unique_together = ('name', 'site',)

    def __str__(self):
        return f"{self.name} ({self.site})"
    
class ProductClick(models.Model):
    product_name = models.CharField(max_length=255)
    site = models.CharField(max_length=255)
    country = models.CharField(max_length=100, blank=True, null=True)
    device = models.CharField(max_length=255, blank=True, null=True)
    timestamp = models.DateTimeField(auto_now_add=True)

    class Meta:
        permissions = [
            ("view_analytics", "Can view analytics"),
        ]

    def __str__(self):
        return f"{self.product_name} clicked on {self.site}"
    

class ActiveScraper(models.Model):
    """
    Tracks currently running background scraping jobs. 
    This replaces the in-memory dict and provides process-safe status tracking.
    """
    query = models.CharField(max_length=255, db_index=True)
    site = models.CharField(max_length=50)
    job_id = models.CharField(max_length=100, unique=True, editable=False)
    start_time = models.DateTimeField(default=now)
    end_time = models.DateTimeField(null=True, blank=True)
    status = models.CharField(
        max_length=20, 
        default='RUNNING',
        choices=[
            ('RUNNING', 'Running'),
            ('SUCCESS', 'Success'),
            ('FAILED', 'Failed'),
            ('EXPIRED', 'Expired/Stale')
        ]
    )

    class Meta:
        # Prevent two different jobs from running for the exact same query/site
        unique_together = ('query', 'site')
        verbose_name = "Active Scraper Job"
        verbose_name_plural = "Active Scraper Jobs"

    def __str__(self):
        return f"[{self.status}] {self.site} for '{self.query}' (Started: {self.start_time.strftime('%H:%M:%S')})"
