import requests
from bs4 import BeautifulSoup
from django.utils.timezone import now
import time
import re
import threading
from queue import Queue

BASE_URL = "https://www.jumia.com.ng"

def parse_price(price_raw: str) -> int:
    """
    Extracts the right-most / maximum price from Jumia price strings.
    Examples:
        "₦ 8,000 - ₦ 37,000" -> 37000
        "₦ 37,000"           -> 37000
    """
    if not price_raw:
        return 0

    # Find all numeric groups
    matches = re.findall(r'[\d,]+', price_raw)
    if not matches:
        return 0

    # Convert all to int (remove commas)
    numbers = [int(m.replace(',', '')) for m in matches]

    # Always take the last (right-hand side) value
    return numbers[-1]

def scrape_jumia_batch(query, batch_size=5, max_products=100, batch_callback=None):
    """
    Scrape Jumia and return results in batches to reduce waiting time.
    
    Args:
        query: Search query
        batch_size: Number of products per batch
        max_products: Maximum products to scrape
        batch_callback: Function to call with each batch of products
    """
    start_time = time.time()
    headers = {
        "User-Agent": "Mozilla/5.0"
    }

    search_url = f"{BASE_URL}/catalog/?q={query.replace(' ', '+')}"
    
    try:
        response = requests.get(search_url, headers=headers, timeout=10)
        if response.status_code != 200:
            print("Failed to fetch Jumia search results.")
            if batch_callback:
                batch_callback([], completed=True, error="Failed to fetch results")
            return []
    except requests.RequestException as e:
        print(f"Request error: {e}")
        if batch_callback:
            batch_callback([], completed=True, error=str(e))
        return []

    soup = BeautifulSoup(response.text, 'html.parser')
    all_products = []

    product_cards = soup.select('article.prd')[:max_products]
    print(f"Found {len(product_cards)} products on Jumia")

    # Process in batches
    for i in range(0, len(product_cards), batch_size):
        batch_cards = product_cards[i:i + batch_size]
        batch_products = []
        
        for card in batch_cards:
            try:
                name_elem = card.select_one('h3.name')
                link_elem = card.select_one('a.core')
                price_elem = card.select_one('div.prc')
                image_tag = card.select_one('img')

                if not all([name_elem, link_elem, price_elem, image_tag]):
                    continue

                name = name_elem.text.strip()
                url = BASE_URL + link_elem['href']
                price_raw = price_elem.get_text(strip=True)
                price = parse_price(price_raw)
                image_url = image_tag.get('data-src') or image_tag.get('src')

                product_data = {
                    'name': name,
                    'site': 'jumia',
                    'url': url,
                    'price': price,
                    'image_url': image_url,
                    'stock': True,
                    'return_policy': "7-day return"
                }
                batch_products.append(product_data)
                
            except Exception as e:
                print(f"Error parsing product card: {e}")
                continue

        all_products.extend(batch_products)
        
        # Send batch to callback if provided
        if batch_callback:
            is_completed = (i + batch_size >= len(product_cards))
            batch_callback(batch_products, completed=is_completed)
        
        # Small delay to simulate progressive loading
        time.sleep(0.1)

    end_time = time.time()
    print(f"✨ Jumia batch scraping finished in {end_time - start_time:.2f} seconds. Found {len(all_products)} products.")
    
    return all_products

def scrape_jumia_async(query, batch_size=5, max_products=30):
    """
    Non-blocking version that uses threading to return batches immediately.
    Returns a generator that yields batches of products.
    """
    def scraper_thread(query, batch_size, max_products, queue):
        try:
            def batch_handler(batch, completed=False, error=None):
                queue.put({
                    'batch': batch,
                    'completed': completed,
                    'error': error
                })
            
            scrape_jumia_batch(query, batch_size, max_products, batch_handler)
        except Exception as e:
            queue.put({
                'batch': [],
                'completed': True,
                'error': str(e)
            })
    
    queue = Queue()
    thread = threading.Thread(
        target=scraper_thread, 
        args=(query, batch_size, max_products, queue),
        daemon=True
    )
    thread.start()
    
    return queue

# Original function maintained for backward compatibility
def scrape_jumia(query):
    """Original synchronous version - returns all products at once"""
    return scrape_jumia_batch(query, batch_size=100, max_products=100)
