# products/consumers.py
import json
from channels.generic.websocket import WebsocketConsumer
from asgiref.sync import async_to_sync
import re
# from .tasks import safe_group_name

class ScraperConsumer(WebsocketConsumer):

    def safe_group_name(self,query: str) -> str:
        """Generate a safe group name for Channels (max 100 chars)."""
        # Replace all non-allowed characters with underscores
        safe = re.sub(r'[^\w\.-]', '_', query)
        # Limit length to 90 (to allow prefix)
        return f"scraper_updates_{safe[:90]}"
    
    def connect(self):
        # 1. Get the 'query' from the URL route kwargs (e.g., /ws/scraper/crane/)
        # self.query = self.scope['url_route']['kwargs']['query'] 
        # print(f"🔥 CONSUMER CONNECTED for query: {self.query}")
        # 2. Define the group name for this specific job/query
        # self.group_name = f'scraper_updates_{self.query}'

        raw_query = self.scope['url_route']['kwargs']['query']
        print(f"🔥 CONSUMER CONNECTED for query: {raw_query}")
        self.query = raw_query
        self.group_name = self.safe_group_name(raw_query)
        print(f"🔥 CONSUMER GROUP NAME for query: {self.group_name}")
        # 3. Add this connection to the group. 
        # The Celery task will send messages to this group.
        async_to_sync(self.channel_layer.group_add)(
            self.group_name,
            self.channel_name
        )
        self.accept()
        print(f"WS connected and joined group: {self.group_name}")

    def disconnect(self, close_code):
        # Remove this connection from the group when disconnected
        async_to_sync(self.channel_layer.group_discard)(
            self.group_name,
            self.channel_name
        )
        print(f"WS disconnected from group: {self.group_name}")

    # Method that receives a message when Celery calls group_send with 'type': 'scraper.update'
    def scraper_update(self, event):
        print(f"🔥 CONSUMER RECEIVED SIGNAL for query: {event.get('query')}. Sending to WebSocket.")
        # Send the received data directly to the frontend WebSocket
        self.send(text_data=json.dumps({
            'type': event.get('status', 'update'), # 'completed'
            'site': event.get('site'),
            'query': event.get('query'),
            'message': event.get('message', 'Scraping status update.')
        }))