from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status, permissions
from rest_framework.authtoken.models import Token
from django.shortcuts import get_object_or_404

from .serializers import RegisterSerializer, LoginSerializer
from   .utils.activity_logger import log_activity

from rest_framework.permissions import IsAuthenticated
from .permissions.is_admin_or_analyst import IsAdminOrAnalyst
from .permissions import IsAdmin
from .utils.activity_logger import log_activity
from .models import ActivityLog
from .serializers import ActivityLogSerializer, AdminUserSerializer

from django.contrib.auth import get_user_model
User = get_user_model()
from django.utils import timezone
from datetime import timedelta
from django.db.models import Count
from products.models import ProductClick
from products.serializers import ProductClickSerializer

from rest_framework.permissions import IsAuthenticated




class RegisterAPIView(APIView):
    authentication_classes = []
    permission_classes = [permissions.AllowAny]

    def post(self, request):
        serializer = RegisterSerializer(data=request.data)
        if serializer.is_valid():
            user = serializer.save()
            token = Token.objects.create(user=user)
            return Response({
                "token": token.key,
                "user": {
                    "username": user.username,
                    "role": user.role
                }
            }, status=status.HTTP_201_CREATED)

        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class LoginAPIView(APIView):
    permission_classes = [permissions.AllowAny]

    def post(self, request):
        serializer = LoginSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        user = serializer.validated_data['user']
        token, _ = Token.objects.get_or_create(user=user)

        log_activity(
            user=user,
            action="login",
            description="User logged in",
            request=request
        )

        return Response({
            "token": token.key,
            "user": {
                "id": user.id,
                "username": user.username,
                "role": user.role
            }
        })


class LogoutAPIView(APIView):
    permission_classes = [permissions.IsAuthenticated]

    def post(self, request):
        log_activity(
            user=request.user,
            action="logout",
            description="User logged out",
            request=request
        )
        request.auth.delete()
        return Response({"detail": "Logged out successfully"})




class ClickAnalyticsView(APIView):
    permission_classes = [IsAuthenticated, IsAdminOrAnalyst]

    def get(self, request):
        log_activity(
            user=request.user,
            action="analytics",
            description="Viewed analytics dashboard",
            request=request
        )

        return Response({"data": "Analytics data"})
    
class ActivityLogAPIView(APIView):
    permission_classes = [IsAdmin]

    def get(self, request):
        logs = ActivityLog.objects.all().order_by("-created_at")[:100]  # latest 100
        serializer = ActivityLogSerializer(logs, many=True)
        return Response(serializer.data)

class PopularSearchTrendsAPIView(APIView):
    permission_classes = [IsAuthenticated, IsAdminOrAnalyst]

    def get(self, request):
        period = request.GET.get("period", "7d")

        now = timezone.now()

        if period == "today":
            start = now.replace(hour=0, minute=0, second=0)
        elif period == "30d":
            start = now - timedelta(days=30)
        else:  # default 7d
            start = now - timedelta(days=7)

        trends = (
            ActivityLog.objects
            .filter(
                action="search",
                created_at__gte=start,
                keyword__isnull=False
            )
            .values("keyword")
            .annotate(count=Count("id"))
            .order_by("-count")[:20]
        )

        return Response({
            "period": period,
            "trends": trends
        })


class AdminUserListAPIView(APIView):
    permission_classes = [IsAdmin]

    def get(self, request):
        users = User.objects.all().order_by("-date_joined")
        serializer = AdminUserSerializer(users, many=True)
        return Response(serializer.data)

class AdminUserUpdateAPIView(APIView):
    permission_classes = [IsAdmin]

    def patch(self, request, user_id):
        user = get_object_or_404(User, id=user_id)

        role = request.data.get("role")
        is_active = request.data.get("is_active")

        if role and role in dict(User.ROLE_CHOICES):
            user.role = role

        if is_active is not None:
            user.is_active = is_active

        user.save()
        return Response({"message": "User updated successfully"})


class ProductClickListAPIView(APIView):
    permission_classes = [IsAuthenticated, IsAdminOrAnalyst]
    """
    API endpoint to fetch all product clicks.
    """

    def get(self, request, format=None):
        clicks = ProductClick.objects.all().order_by('-timestamp')  
        serializer = ProductClickSerializer(clicks, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)